<?php
session_start();
require_once "../includes/role_helpers.php";
requireAdmin();
require_once "../config/config.php";

$conn = getDbConnection();

$success_msg = $error_msg = "";

// Predefined categories
$predefined_categories = [
    "Anti hypertensive", "Anti diabetic", "Antibiotic", "Anti malaria",
    "Psychotics", "Cough Mixtures", "Blood Tonics", "Multi Vitamins",
    "Supplements", "Cosmetics", "Lozenges", "Analgesics", "Anti Allergy"
];

// Ensure predefined categories are in DB
foreach ($predefined_categories as $cat) {
    $stmt = $conn->prepare("SELECT category_id FROM categories WHERE category_name = ?");
    $stmt->bind_param("s", $cat);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($result->num_rows === 0) {
        $ins = $conn->prepare("INSERT INTO categories (category_name) VALUES (?)");
        $ins->bind_param("s", $cat);
        $ins->execute();
        $ins->close();
    }
    $stmt->close();
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $category_name = $_POST['category_name'] ?? '';
    if ($category_name) {
        $stmt = $conn->prepare("INSERT INTO categories (category_name) VALUES (?)");
        $stmt->bind_param("s", $category_name);
        if ($stmt->execute()) {
            $success_msg = "Category added.";
        } else {
            $error_msg = "Error adding category.";
        }
        $stmt->close();
    } else {
        $error_msg = "Choose a category.";
    }
}

if (isset($_GET['delete'])) {
    $del_id = intval($_GET['delete']);
    $stmt = $conn->prepare("DELETE FROM categories WHERE category_id = ?");
    $stmt->bind_param("i", $del_id);
    $stmt->execute();
    $stmt->close();
    $success_msg = "Category deleted.";
}

$categories = $conn->query("SELECT category_id, category_name FROM categories ORDER BY category_name");

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Categories - Admin</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" />
    <style>
        body {
            background: #f4f7fa;
            font-family: "Poppins", sans-serif;
            min-height: 100vh;
            padding: 2rem;
        }
        .container {
            max-width: 700px;
            background: white;
            padding: 2rem 3rem;
            border-radius: 12px;
            box-shadow: 0 8px 40px rgb(0 0 0 / 0.12);
            margin: auto;
        }
        h2 {
            color: #17a2b8;
            text-align: center;
            margin-bottom: 1.5rem;
            font-weight: 700;
        }
        .alert {
            font-weight: 600;
            border-radius: 8px;
        }
        label {
            font-weight: 600;
            color: #495057;
        }
        select.form-select {
            cursor: pointer;
            border-radius: 8px;
            height: 42px;
            padding: 0 1rem;
            font-weight: 600;
        }
        button.btn-primary {
            background-color: #17a2b8;
            border-color: #17a2b8;
            font-weight: 700;
            border-radius: 8px;
            transition: background-color 0.3s ease;
        }
        button.btn-primary:hover {
            background-color: #138496;
            border-color: #117a8b;
        }
        table {
            margin-top: 2rem;
        }
        thead th {
            background: #17a2b8;
            color: white;
            font-weight: 600;
            text-align: center;
            border: none;
            border-radius: 8px 8px 0 0;
            padding: 12px;
        }
        tbody tr {
            border-bottom: 1px solid #dee2e6;
        }
        tbody td {
            vertical-align: middle;
            padding: 12px 15px;
            text-align: center;
        }
        a.btn-danger {
            padding: 4px 10px;
            font-size: 0.9rem;
            border-radius: 6px;
        }
    </style>
</head>
<body>
<div class="container">
    <h2>Category Management</h2>

    <?php if ($success_msg): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success_msg) ?></div>
    <?php elseif ($error_msg): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error_msg) ?></div>
    <?php endif; ?>

    <form method="post" class="mb-4" autocomplete="off" novalidate>
        <div class="mb-3">
            <label for="category_name">Choose Category</label>
            <select id="category_name" name="category_name" class="form-select" required>
                <option value="" disabled selected>-- Select a Category --</option>
                <?php foreach ($predefined_categories as $cat): ?>
                    <option value="<?= htmlspecialchars($cat) ?>"><?= htmlspecialchars($cat) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <button type="submit" class="btn btn-primary w-100">Add Category</button>
    </form>

    <table class="table table-striped table-hover rounded shadow-sm">
        <thead>
            <tr>
                <th>Category Name</th>
                <th>Action</th>
            </tr>
        </thead>
        <tbody>
            <?php while ($cat = $categories->fetch_assoc()): ?>
                <tr>
                    <td><?= htmlspecialchars($cat['category_name']) ?></td>
                    <td>
                        <a href="?delete=<?= $cat['category_id'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('Delete this category?');">Delete</a>
                    </td>
                </tr>
            <?php endwhile; ?>
        </tbody>
    </table>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
