<?php
session_start();
require_once "../includes/role_helpers.php";
requireAdmin();
require_once "../config/config.php";

$conn = getDbConnection();

$error_msg = "";
$success_msg = "";
$product_id = intval($_GET['id'] ?? 0);

if ($product_id <= 0) {
    die("Invalid product ID.");
}

// Fetch categories for select dropdown
$categories_result = $conn->query("SELECT category_name FROM categories ORDER BY category_name");

// Fetch existing product data including barcode and category string
$stmt = $conn->prepare("SELECT product_name, price, category, barcode FROM products WHERE product_id = ?");
$stmt->bind_param("i", $product_id);
$stmt->execute();
$stmt->bind_result($product_name, $price, $category, $barcode);
if (!$stmt->fetch()) {
    die("Product not found.");
}
$stmt->close();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $product_name = trim($_POST['product_name'] ?? '');
    $price_input = $_POST['price'] ?? '';
    $category_input = trim($_POST['category'] ?? '');
    $barcode_input = trim($_POST['barcode'] ?? '');

    // Validation
    if ($product_name === '') {
        $error_msg = "Product name is required.";
    } elseif (!is_numeric($price_input) || floatval($price_input) <= 0) {
        $error_msg = "Price must be a valid number greater than zero.";
    } elseif ($category_input === '') {
        $error_msg = "Please select a valid category.";
    } else {
        $price_val = floatval($price_input);
        // Barcode optional: set null if empty
        $barcode_val = $barcode_input !== '' ? $barcode_input : null;

        $update_stmt = $conn->prepare("UPDATE products SET product_name = ?, price = ?, category = ?, barcode = ?, updated_at = CURRENT_TIMESTAMP WHERE product_id = ?");
        $update_stmt->bind_param("sdssi", $product_name, $price_val, $category_input, $barcode_val, $product_id);
        if ($update_stmt->execute()) {
            $success_msg = "Product updated successfully.";
            $category = $category_input; // update for redisplay
            $barcode = $barcode_val;
        } else {
            $error_msg = "Error updating product: " . $update_stmt->error;
        }
        $update_stmt->close();
    }
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Edit Product</title>
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap-icons/1.13.0/font/bootstrap-icons.min.css" rel="stylesheet" />
</head>
<body>
<div class="container bg-white p-4 rounded shadow-sm mt-4" style="max-width:600px;">
    <h2>Edit Product</h2>

    <?php if ($success_msg): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success_msg) ?></div>
    <?php endif; ?>
    <?php if ($error_msg): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error_msg) ?></div>
    <?php endif; ?>

    <form method="post" novalidate>
        <div class="mb-3">
            <label for="product_name" class="form-label">Product Name *</label>
            <input type="text" id="product_name" name="product_name" class="form-control <?= $error_msg && $product_name === '' ? 'is-invalid' : '' ?>" value="<?= htmlspecialchars($product_name) ?>" required />
        </div>

        <div class="mb-3">
            <label for="price" class="form-label">Price (GH₵) *</label>
            <input type="number" step="0.01" min="0.01" id="price" name="price" class="form-control <?= $error_msg && !is_numeric($price) ? 'is-invalid' : '' ?>" value="<?= htmlspecialchars($price) ?>" required />
        </div>

        <div class="mb-3">
            <label for="category" class="form-label">Category *</label>
            <select id="category" name="category" class="form-select <?= $error_msg && $category === '' ? 'is-invalid' : '' ?>" required>
                <option value="">Select category</option>
                <?php while ($cat = $categories_result->fetch_assoc()): ?>
                    <option value="<?= htmlspecialchars($cat['category_name']) ?>" <?= $category === $cat['category_name'] ? 'selected' : '' ?>>
                        <?= htmlspecialchars($cat['category_name']) ?>
                    </option>
                <?php endwhile; ?>
            </select>
        </div>

        <div class="mb-3">
            <label for="barcode" class="form-label">Barcode (optional)</label>
            <input type="text" id="barcode" name="barcode" class="form-control" value="<?= htmlspecialchars($barcode ?? '') ?>" />
        </div>

        <div class="d-flex gap-2">
            <button type="submit" class="btn btn-primary"><i class="bi bi-save"></i> Save Changes</button>
            <a href="inventory.php" class="btn btn-secondary"><i class="bi bi-x-circle"></i> Cancel</a>
        </div>
    </form>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
