<?php
require_once "../config/config.php";
session_start();
require_once "../includes/role_helpers.php";
requireLogin();

$conn = getDbConnection();

// Pagination parameters
$page = max(1, intval($_GET['page'] ?? 1));
$per_page = 15;
$offset = ($page - 1) * $per_page;

// Handle mark-as-read POST action (for non-AJAX fallback)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['mark_read_id'])) {
    $alert_id = intval($_POST['mark_read_id']);
    $stmt = $conn->prepare("UPDATE stock_alerts SET is_read = TRUE WHERE alert_id = ?");
    $stmt->bind_param("i", $alert_id);
    $stmt->execute();
    $stmt->close();
    header("Location: inventory_alerts.php?page=$page");
    exit;
}

// Count total alerts
$total_res = $conn->query("SELECT COUNT(*) AS total FROM stock_alerts");
$total_alerts = $total_res->fetch_assoc()['total'] ?? 0;
$total_pages = ceil($total_alerts / $per_page);

// Fetch alerts with joins to stock and medicines for names
$sql = "
SELECT sa.alert_id, sa.alert_type, sa.message, sa.is_read, sa.created_at, 
       s.quantity, s.expiry_date, 
       m.name AS medicine_name
FROM stock_alerts sa
JOIN stock s ON s.stock_id = sa.stock_id
JOIN medicines m ON s.medicine_id = m.medicine_id
ORDER BY sa.created_at DESC
LIMIT $offset, $per_page
";

$stmt = $conn->prepare($sql);
if (!$stmt) {
    die("MySQL prepare error: " . $conn->error);
}

$stmt->execute();
$result = $stmt->get_result();

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<title>Inventory Alerts - Adwenpa Pharma</title>
<meta name="viewport" content="width=device-width, initial-scale=1" />
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
</head>
<body>
<div class="container my-4">
    <a href="dashboard.php" class="btn btn-secondary mb-3">&larr; Back to Dashboard</a>
    <h2>Inventory Alerts</h2>
    <table class="table table-striped table-bordered align-middle">
        <thead class="table-dark">
            <tr>
                <th>#</th>
                <th>Medicine</th>
                <th>Alert Type</th>
                <th>Message</th>
                <th>Quantity</th>
                <th>Expiry Date</th>
                <th>Created At</th>
                <th>Status</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
        <?php 
        if ($result->num_rows === 0): ?>
            <tr><td colspan="9" class="text-center">No alerts found.</td></tr>
        <?php else: 
            $count = $offset + 1;
            while($alert = $result->fetch_assoc()): ?>
            <tr class="<?= $alert['is_read'] ? '' : 'table-warning' ?>">
                <td><?= $count++ ?></td>
                <td><?= htmlspecialchars($alert['medicine_name']) ?></td>
                <td><?= ucfirst(str_replace('_', ' ', $alert['alert_type'])) ?></td>
                <td><?= htmlspecialchars($alert['message']) ?></td>
                <td><?= (int)$alert['quantity'] ?></td>
                <td><?= htmlspecialchars($alert['expiry_date']) ?></td>
                <td><?= htmlspecialchars($alert['created_at']) ?></td>
                <td class="status-cell"><?= $alert['is_read'] ? 'Read' : 'Unread' ?></td>
                <td>
                    <?php if (!$alert['is_read']): ?>
                    <button class="btn btn-sm btn-primary mark-read-btn" data-alert-id="<?= $alert['alert_id'] ?>">Mark Read</button>
                    <?php else: ?>
                    <span class="text-muted">-</span>
                    <?php endif; ?>
                </td>
            </tr>
        <?php endwhile; 
        endif; ?>
        </tbody>
    </table>

    <!-- Pagination -->
    <nav aria-label="Page navigation">
        <ul class="pagination justify-content-center">
        <?php if ($page > 1): ?>
            <li class="page-item"><a class="page-link" href="?page=<?= $page-1 ?>">Previous</a></li>
        <?php else: ?>
            <li class="page-item disabled"><span class="page-link">Previous</span></li>
        <?php endif; ?>

        <?php for ($i=1; $i <= $total_pages; $i++): ?>
            <li class="page-item<?= $i === $page ? ' active' : '' ?>">
                <a class="page-link" href="?page=<?= $i ?>"><?= $i ?></a>
            </li>
        <?php endfor; ?>

        <?php if ($page < $total_pages): ?>
            <li class="page-item"><a class="page-link" href="?page=<?= $page+1 ?>">Next</a></li>
        <?php else: ?>
            <li class="page-item disabled"><span class="page-link">Next</span></li>
        <?php endif; ?>
        </ul>
    </nav>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<script>
document.querySelectorAll('.mark-read-btn').forEach(btn => {
    btn.addEventListener('click', function() {
        const alertId = this.dataset.alertId;
        fetch('mark_alert_read.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
            body: `alert_id=${alertId}`
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const row = btn.closest('tr');
                row.classList.remove('table-warning');
                row.querySelector('.status-cell').textContent = 'Read';
                btn.remove();
                const span = document.createElement('span');
                span.classList.add('text-muted');
                span.textContent = '-';
                row.querySelector('td:last-child').appendChild(span);
            } else {
                alert('Failed to mark alert as read.');
            }
        });
    });
});
</script>
</body>
</html>
