<?php
session_start();
require_once "../includes/role_helpers.php";
require_once "../config/config.php";
require_once "../includes/logging.php";  // Include logging helper

if (isset($_SESSION['loggedin']) && $_SESSION['loggedin'] === true) {
    header("Location: dashboard.php");
    exit;
}

$username = $password = "";
$username_err = $password_err = $login_err = "";

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $username = trim($_POST["username"] ?? "");
    $password = trim($_POST["password"] ?? "");

    if (empty($username)) $username_err = "Please enter username.";
    if (empty($password)) $password_err = "Please enter your password.";

    if (empty($username_err) && empty($password_err)) {
        $conn = getDbConnection();

        $stmt = $conn->prepare("SELECT user_id, username, password_hash, role, branch_id FROM users WHERE username=? LIMIT 1");
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $stmt->store_result();

        if ($stmt->num_rows == 1) {
            $stmt->bind_result($user_id, $user_name, $hashed_password, $role, $user_branch_id);
            $stmt->fetch();

            if (password_verify($password, $hashed_password)) {
                // Successful login
                $_SESSION['user_id'] = $user_id;
                $_SESSION['username'] = $user_name;
                $_SESSION['role'] = $role;
                $_SESSION['loggedin'] = true;

                // Admin user gets full access to all branches
                if ($role === 'admin') {
                    $_SESSION['branch_id'] = 0; // 0 means all branches
                } else {
                    $_SESSION['branch_id'] = $user_branch_id;
                }

                // Log successful login
                log_action($conn, $user_id, 'User logged in', "Username: $user_name");

                header("Location: dashboard.php");
                exit;
            } else {
                // Log failed login attempt (invalid password)
                log_action($conn, 0, 'Failed login attempt', "Username: $username, Wrong password");
                $login_err = "Invalid username or password.";
            }
        } else {
            // Log failed login attempt (username not found)
            log_action($conn, 0, 'Failed login attempt', "Username not found: $username");
            $login_err = "Invalid username or password.";
        }
        $stmt->close();
        $conn->close();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Login - Adwenpa Pharma</title>
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
    <style>
        body, html {
            height: 100%;
            background: linear-gradient(135deg, #4a8ef1, #134e90);
        }
        .login-container {
            max-width: 400px;
            margin: auto;
            margin-top: 8vh;
            padding: 30px;
            background-color: #ffffffdd;
            border-radius: 10px;
            box-shadow: 0 12px 24px rgba(18, 38, 63, 0.3);
        }
        .login-title {
            font-weight: 700;
            margin-bottom: 25px;
            color: #134e90;
            text-align: center;
            letter-spacing: 1px;
        }
        .btn-primary {
            background-color: #134e90;
            border: none;
            transition: background-color 0.3s ease;
        }
        .btn-primary:hover {
            background-color: #0a3563;
        }
        .form-label {
            font-weight: 600;
        }
    </style>
</head>
<body>
<div class="login-container">
    <h2 class="login-title">Welcome to Adwenpa Pharma</h2>
    <p class="text-center mb-4">Please fill in your credentials to login.</p>

    <?php if ($login_err): ?>
        <div class="alert alert-danger" role="alert"><?= htmlspecialchars($login_err) ?></div>
    <?php endif; ?>

    <form action="login.php" method="post" novalidate>
        <div class="mb-3">
            <label for="username" class="form-label">Username</label>
            <input 
                type="text" 
                id="username" 
                name="username" 
                class="form-control <?= $username_err ? 'is-invalid' : ''; ?>" 
                value="<?= htmlspecialchars($username) ?>" 
                autofocus 
                required 
            />
            <div class="invalid-feedback"><?= htmlspecialchars($username_err) ?></div>
        </div>
        <div class="mb-4">
            <label for="password" class="form-label">Password</label>
            <input 
                type="password" 
                id="password" 
                name="password" 
                class="form-control <?= $password_err ? 'is-invalid' : ''; ?>" 
                required 
            />
            <div class="invalid-feedback"><?= htmlspecialchars($password_err) ?></div>
        </div>
        <button type="submit" class="btn btn-primary w-100">Login</button>
    </form>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
