<?php
require_once "../config/config.php";
session_start();
require_once "../includes/role_helpers.php";
requireLogin();

$conn = getDbConnection();

if (!isset($_SESSION['cart'])) {
    $_SESSION['cart'] = [];
}

$error = null;

// Fetch units for dropdown
$units_result = $conn->query("SELECT unit_name FROM units ORDER BY unit_name");
$units = $units_result ? $units_result->fetch_all(MYSQLI_ASSOC) : [];

// Handle add to cart
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_product_id'], $_POST['quantity'], $_POST['unit'])) {
        $pid = intval($_POST['add_product_id']);
        $qty = max(1, intval($_POST['quantity']));
        $unit = trim($_POST['unit']);
        if (isset($_SESSION['cart'][$pid])) {
            $_SESSION['cart'][$pid]['quantity'] += $qty;
            $_SESSION['cart'][$pid]['unit'] = $unit;
        } else {
            $_SESSION['cart'][$pid] = ['quantity' => $qty, 'unit' => $unit];
        }
        header("Location: pos.php");
        exit;
    }
    // ... Update cart and checkout code unchanged ...
}

$products_result = $conn->query("SELECT product_id, product_name, barcode, price FROM products ORDER BY product_name ASC");
$products = [];
while ($pr = $products_result->fetch_assoc()) {
    $products[$pr['product_id']] = $pr;
}

$cart_items = [];
if (!empty($_SESSION['cart'])) {
    $ids = implode(',', array_map('intval', array_keys($_SESSION['cart'])));
    $sql = "SELECT product_id, product_name, price FROM products WHERE product_id IN ($ids)";
    $result = $conn->query($sql);
    while ($row = $result->fetch_assoc()) {
        $product_id = $row['product_id'];
        $qty = $_SESSION['cart'][$product_id]['quantity'];
        $unit = $_SESSION['cart'][$product_id]['unit'];
        $cart_items[$product_id] = [
            'product_name' => $row['product_name'],
            'price' => $row['price'],
            'quantity' => $qty,
            'unit' => $unit,
            'subtotal' => $row['price'] * $qty,
        ];
    }
}

$total = 0;
foreach ($cart_items as $item) {
    $total += $item['subtotal'];
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>POS - Adwenpa Pharma</title>
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
</head>
<body>
<nav class="navbar navbar-expand-lg navbar-dark bg-primary">
    <div class="container-fluid">
        <a class="navbar-brand" href="dashboard.php">Adwenpa Pharma</a>
        <div class="d-flex">
            <a href="dashboard.php" class="btn btn-secondary me-2">Back to Dashboard</a>
            <a href="logout.php" class="btn btn-outline-light btn-sm">Logout</a>
        </div>
    </div>
</nav>

<div class="container my-4">
    <h2>Point of Sale</h2>

    <?php if ($error): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
    <?php endif; ?>

    <div class="row g-3 mb-3 align-items-center">
        <div class="col-auto">
            <label for="barcode_input" class="col-form-label">Scan Barcode</label>
        </div>
        <div class="col-auto" style="max-width: 250px;">
            <input type="text" id="barcode_input" class="form-control" placeholder="Scan barcode here" autofocus autocomplete="off" />
        </div>
    </div>

    <form method="post" class="row g-3 mb-4 align-items-center">
        <div class="col-auto">
            <label for="add_product_id" class="col-form-label">Product</label>
        </div>
        <div class="col-auto">
            <select name="add_product_id" id="add_product_id" class="form-select" required>
                <option value="" disabled selected>Select product</option>
                <?php foreach ($products as $id => $p): ?>
                    <option value="<?= $id ?>"><?= htmlspecialchars($p['product_name']) ?> - ₵<?= number_format($p['price'], 2) ?></option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="col-auto">
            <label for="unit" class="col-form-label">Unit</label>
        </div>
        <div class="col-auto">
            <select name="unit" id="unit" class="form-select" required>
                <option value="" disabled selected>Select unit</option>
                <?php foreach ($units as $unit_item):
                    $selected = (isset($_POST['unit']) && $_POST['unit'] === $unit_item['unit_name']) ||
                                (!isset($_POST['unit']) && isset($_SESSION['cart']) && !empty($_SESSION['cart']) &&
                                  $_SESSION['cart'][array_key_first($_SESSION['cart'])]['unit'] === $unit_item['unit_name']) ? 'selected' : '';
                ?>
                    <option value="<?= htmlspecialchars($unit_item['unit_name']) ?>" <?= $selected ?>><?= htmlspecialchars($unit_item['unit_name']) ?></option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="col-auto">
            <label for="quantity" class="col-form-label">Quantity</label>
        </div>
        <div class="col-auto">
            <input type="number" name="quantity" id="quantity" value="1" min="1" class="form-control" required />
        </div>

        <div class="col-auto">
            <button type="submit" class="btn btn-primary">Add to Cart</button>
        </div>
    </form>

    <?php if (empty($cart_items)): ?>
        <div class="alert alert-info">Your cart is empty.</div>
    <?php else: ?>
        <form method="post">
            <table class="table table-bordered align-middle text-center">
                <thead class="table-light">
                    <tr>
                        <th>Product</th>
                        <th>Unit</th>
                        <th>Price (₵)</th>
                        <th>Quantity</th>
                        <th>Subtotal (₵)</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($cart_items as $pid => $item): ?>
                        <tr>
                            <td><?= htmlspecialchars($item['product_name']) ?></td>
                            <td><?= htmlspecialchars($item['unit']) ?></td>
                            <td><?= number_format($item['price'], 2) ?></td>
                            <td>
                                <input type="number" name="quantities[<?= $pid ?>]" value="<?= $item['quantity'] ?>" min="0" class="form-control form-control-sm" />
                            </td>
                            <td><?= number_format($item['subtotal'], 2) ?></td>
                        </tr>
                    <?php endforeach; ?>
                    <tr class="fw-bold">
                        <td colspan="4" class="text-end">Total</td>
                        <td>₵<?= number_format($total, 2) ?></td>
                    </tr>
                </tbody>
            </table>

            <div class="d-flex justify-content-between">
                <button type="submit" name="update_cart" class="btn btn-secondary">Update Cart</button>
                <button type="submit" name="checkout" class="btn btn-success" onclick="return confirm('Confirm checkout?')">Checkout</button>
            </div>
        </form>
    <?php endif; ?>
</div>

<script>
document.getElementById('barcode_input').addEventListener('keydown', function(e) {
    if (e.key === 'Enter') {
        e.preventDefault();
        const barcode = this.value.trim();
        if (barcode.length === 0) return;
        this.value = ''; // clear input

        fetch('pos_barcode_lookup.php?barcode=' + encodeURIComponent(barcode))
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const form = document.createElement('form');
                    form.method = 'POST';
                    form.action = 'pos.php';

                    const inputId = document.createElement('input');
                    inputId.type = 'hidden';
                    inputId.name = 'add_product_id';
                    inputId.value = data.product_id;
                    form.appendChild(inputId);

                    const inputQty = document.createElement('input');
                    inputQty.type = 'hidden';
                    inputQty.name = 'quantity';
                    inputQty.value = 1;
                    form.appendChild(inputQty);

                    document.body.appendChild(form);
                    form.submit();
                } else {
                    alert('Product not found for barcode: ' + barcode);
                }
            })
            .catch(() => {
                alert('Error looking up barcode.');
            });
    }
});
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
