<?php
// Use __DIR__ to build absolute path to config.php for reliability
require_once __DIR__ . "/config/config.php";

$conn = getDbConnection();

$setup_allowed = true;

// Check if any admin exists already
$result = $conn->query("SELECT user_id FROM users WHERE role='admin' LIMIT 1");
if ($result && $result->num_rows > 0) {
    $setup_allowed = false;  // Admin exists, disable setup
}
$setup_message = "";

if (!$setup_allowed) {
    $setup_message = "Admin account already exists. Setup not allowed.";
}

$username = $full_name = $email = $password = "";
$username_err = $email_err = $password_err = "";

if ($_SERVER['REQUEST_METHOD'] === 'POST' && $setup_allowed) {
    $username = trim($_POST['username']);
    $full_name = trim($_POST['full_name']);
    $email = trim($_POST['email']);
    $password = trim($_POST['password']);

    if (empty($username)) { $username_err = "Username required."; }
    if (empty($email)) { $email_err = "Email required."; }
    if (empty($password) || strlen($password) < 6) { $password_err = "Password minimum 6 chars."; }

    if (empty($username_err) && empty($email_err) && empty($password_err)) {
        // Hash the password securely before saving to DB
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);

        $stmt = $conn->prepare("INSERT INTO users (username, full_name, email, password_hash, role) VALUES (?, ?, ?, ?, 'admin')");
        $stmt->bind_param("ssss", $username, $full_name, $email, $hashed_password);
        if ($stmt->execute()) {
            $setup_message = "Admin user created successfully. Please delete this setup file from the server.";
        } else {
            $setup_message = "Error creating admin user: " . $stmt->error;
        }
        $stmt->close();
    }
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Initial Admin Setup - Adwenpa Pharma</title>
    <link rel="stylesheet" href="../public/css/bootstrap.min.css" />
    <style>
        .wrapper { max-width: 450px; margin: 30px auto; }
    </style>
</head>
<body>
<div class="wrapper">
    <h2>Initial Admin Setup</h2>
    <?php if ($setup_message): ?>
        <div class="alert alert-info"><?= htmlspecialchars($setup_message) ?></div>
    <?php endif; ?>
    <?php if ($setup_allowed): ?>
    <form method="post" novalidate>
        <div class="form-group mb-3">
            <label>Username</label>
            <input type="text" name="username" class="form-control <?= $username_err ? 'is-invalid' : '' ?>" value="<?= htmlspecialchars($username) ?>" required />
            <div class="invalid-feedback"><?= htmlspecialchars($username_err) ?></div>
        </div>
        <div class="form-group mb-3">
            <label>Full Name</label>
            <input type="text" name="full_name" class="form-control" value="<?= htmlspecialchars($full_name) ?>" />
        </div>
        <div class="form-group mb-3">
            <label>Email</label>
            <input type="email" name="email" class="form-control <?= $email_err ? 'is-invalid' : '' ?>" value="<?= htmlspecialchars($email) ?>" required />
            <div class="invalid-feedback"><?= htmlspecialchars($email_err) ?></div>
        </div>
        <div class="form-group mb-3">
            <label>Password (Min 6 Characters)</label>
            <input type="password" name="password" class="form-control <?= $password_err ? 'is-invalid' : '' ?>" required />
            <div class="invalid-feedback"><?= htmlspecialchars($password_err) ?></div>
        </div>
        <button type="submit" class="btn btn-primary">Create Admin</button>
    </form>
    <?php endif; ?>
</div>
</body>
</html>
